from matplotlib import pyplot as plt
import numpy as np
from scipy import constants, linalg
import numexpr

__version__ = '0.0'

__doc__ = """
Module for numerically solving the field of a set of conductors at specified
potentials in 2D, i.e. all objects are infinitely extended along the third
dimension. Supports dielectrics. The conductors are approximated as polygons
while the dielectrics as sets of rectangles. On a side of a conductor the
charge density is assumed constant, and on a rectangle of a dielectric the
polarization density is assumed constant.

Algorithm
---------
The charge and polarization densities are obtained by solving a linear system.
The system is generated by imposing the effect of all elementary objects on the
geometrical center of each elementary object. For example: the contribute of
the potential of all conductor segments is computed in the center of each
segment. This means that if one wants to know the potential or field near an
elementary object, she should compute it at the center of the object, otherwise
the approximation that the charge or polarization density is constant on the
elementary object will give significative effects. For example: an uniformly
polarized region is equivalent to a polarization charge on its boundaries, so
near the boundary of an elementary dielectric rectangle the field will be
orthogonal to the sides.

Boundary conditions
-------------------
In 3D one has to impose a value of the potential at infinity to solve the
system. In 2D, since the monopole term of the potential is logarithmically
diverging, either the potential at infinity is diverging or it is zero *and*
the total charge of the set of conductors is zero. This means that if one
specifies the potentials of all conductors and requires zero potential at
infinity the system is overdetermined. When solving the system, the user can
ask to keep the potentials as specified and let the potential diverge, or allow
the potential to shift to have a null monopole.

Units
-----
The units are SI. This is fixed by the value of vacuum permittivity used
internally. If the user wants to input values in different units, she has to
take into account the epsilon_0 factors in the results.

Requirements
------------
matplotlib, numpy, scipy, numexpr.

Classes
-------
Conductor :
    Conductor objects.
Dielectric :
    Dielectric objects.
CircleConductor, SegmentConductor, RectangleConductor, RectangleDielectric :
    Convenience classes for constructing simple geometries.
ConductorSet :
    Class for aggregating objects and solving the system.

Examples
--------
Simulate a cylindric capacitor:
>>> from estatic2d import CircleConductor, ConductorSet
>>> from numpy import linspace, meshgrid, abs
>>>
>>> s = ConductorSet(
>>>         CircleConductor((0, 0), 1, 100, potential=1, name='inner'),
>>>         CircleConductor((0, 0), 2, 100, potential=0, name='outer')
>>>     )
>>> s.solve()
>>>
>>> x = y = linspace(-2, 2, 100)
>>> s.draw() # draw the geometry
>>> s.draw_potential(x, y)
>>> s.draw_field(*meshgrid(x, y))
>>> capacitance_per_unit_length = abs(s.conductors[0].charge_per_unit_length / (s.conductors[0].potential - s.conductors[1].potential))
"""

__all__ = [
    'Conductor',
    'CircleConductor',
    'RectangleConductor',
    'SegmentConductor',
    'Dielectric',
    'RectangleDielectric',
    'ConductorSet'
]

class Conductor(object):
    """
    Class to represent conductors. A conductor is a polygonal chain, eventually
    closed. On each segment the charge density is constant.
    
    Arguments
    ---------
    vertexes_x : array of shape (N,)
    vertexes_y : array of shape (N,)
        x and y coordinates of the vertices of the polygonal chain, in meters.
    closed : boolean
        If True: if the chain specified by vertexes_x and vertexes_y is open,
        close it. If False: if the chain is closed, raise a ValueError.
    potential : number
        The electric potential of the conductor, in volts.
    name : string
        Label used to identify the conductor in plots.
    draw_kwargs : dictionary
        Options passed to matplotlib.pyplot.plot to draw the conductor.
    
    Methods
    -------
    draw :
        Draw the polygonal chain on a matplotlib axis.
    compute_potential :
        Compute the potential generated by the conductor at specified points.
    compute_field :
        Compute the field generated by the conductor at specified points.
    
    Properties
    ----------
    potential :
        Potential of the conductor, in volts.
    lengths :
        Array of the lengths of the segments, in meters.
    length :
        Total length of the polygonal chain, in meters.
    centers :
        Array of shape (2, M) of the coordinates x, y of the centers of the
        segments, in meters.
    slopes :
        Array of shape (2, M) of the cosine and sine of the angles between the
        segments and the x axis.
    sigmas :
        Array of shape (M,) of the charge densities of the segments, in farad
        over meter squared. The density is along the z axis and along the
        segment. It is None if the conductor is not in a solved ConductorSet.
    charge_per_unit_length :
        Total charge density of the conductor, in farad over meter. The density
        is along the z axis.
    
    See also
    --------
    SegmentConductor, CircleConductor, RectangleConductor
    """
    def __init__(self, vertexes_x, vertexes_y, closed=True, potential=0, name='conductor', draw_kwargs=dict()):
        vertexes_x = np.asarray(vertexes_x)
        vertexes_y = np.asarray(vertexes_y)
        
        assert len(vertexes_x.shape) == 1
        assert len(vertexes_y.shape) == 1
        assert len(vertexes_x) == len(vertexes_y)
        
        self.vertexes = np.array([vertexes_x, vertexes_y])
        
        factually_closed = np.all(self.vertexes[:,0] == self.vertexes[:,-1])
        if factually_closed and not closed:
            raise ValueError('The chain is actually closed but closed=False.')
        if closed and not factually_closed:
            self.vertexes = np.concatenate([self.vertexes, self.vertexes[:,0:1]], axis=1)
            
        assert isinstance(name, str)
        assert isinstance(draw_kwargs, dict)
        assert np.isscalar(potential)
        
        self._draw_kwargs = draw_kwargs
        
        self.closed = closed
        self._potential = potential
        self.name = name
        
        self._sigmas = None
    
    @property
    def potential(self):
        return self._potential
    
    @property
    def lengths(self):
        return np.sqrt(np.sum(np.diff(self.vertexes, axis=1) ** 2, axis=0))
    
    @property
    def length(self):
        return np.sum(self.lengths)
    
    @property
    def centers(self):
        return (self.vertexes[:,1:] + self.vertexes[:,:-1]) / 2
    
    @property
    def slopes(self):
        return np.diff(self.vertexes, axis=1) / self.lengths
    
    @property
    def sigmas(self):
        return self._sigmas
    
    @property
    def charge_per_unit_length(self):
        return np.sum(self.sigmas * self.lengths)
        
    def draw(self, ax=None, **kw):
        """
        Draw the polygonal chain on a matplotlib axis. The drawing options are
        taken from the draw_kwargs argument specified at object initialization,
        which takes precedence over the default options.
        
        Arguments
        ---------
        ax : None or matplotlib axis
            If None, use the current axis.
        
        Keyword arguments
        -----------------
        Additional keyword arguments are passed to ax.plot, taking precedence
        over the defaults and over draw_kwargs.
        
        Returns
        -------
        The return value of ax.plot.
        
        See also
        --------
        Dielectric.draw, ConductorSet.draw
        """
        if ax is None:
            ax = plt.gca()
        kwargs = dict(
            marker='.',
            markersize=4,
            label='{:s} V = {:.2g}'.format(self.name, self.potential)
        )
        kwargs.update(self._draw_kwargs)
        kwargs.update(kw)
        return ax.plot(self.vertexes[0], self.vertexes[1], **kwargs)
    
    def compute_potential(self, x, y):
        """
        Compute the potential generated by the conductor at specified points.
        
        Arguments
        ---------
        x, y: arrays with same shape
            x and y coordinates of the points, in meters.
        
        Returns
        -------
        Array of potentials, in volts, with same shape as x and y.
        
        See also
        --------
        Dielectric.compute_potential, ConductorSet.compute_potential
        """
        where = np.array([x, y])
        where_shape = (1,) * (len(where.shape) - 1)
        
        l = self.lengths
        l = 1/2 * np.array([-l, l])
        l =           l.reshape(2, *where_shape,    -1)
        m = self.slopes
        mx, my =   m.reshape(2, 1, *where_shape,    -1)
        c = self.centers
        cx, cy =   c.reshape(2, 1, *where_shape,    -1)
        x, y = where.reshape(2, 1, *where.shape[1:], 1)

        integrals = numexpr.evaluate('-1/2 * (\
        2 / (mx ** 2 + my ** 2) * abs(mx * (y - cy) - my * (x - cx)) *\
        arctan(((mx ** 2 + my ** 2) * l + mx * (cx - x) + my * (cy - y)) / abs(mx * (y - cy) - my * (x - cx))) -\
        2 * l +\
        ((mx * (cx - x) + my * (cy - y)) / (mx ** 2 + my ** 2) + l) *\
        log((mx * l - (x - cx)) ** 2 + (my * l - (y - cy)) ** 2))')

        potential = np.sum((integrals[1] - integrals[0]) * self.sigmas.reshape(*where_shape, -1), axis=-1) / (2 * np.pi * constants.epsilon_0)
        
        assert potential.shape == where.shape[1:]

        return potential
    
    def compute_field(self, x, y):
        """
        Compute the electric field generated by the conductor at specified
        points.
        
        Arguments
        ---------
        x, y: arrays with shape S
            x and y coordinates of the points, in meters.
        
        Returns
        -------
        Array with shape (2, *S) of field x and y components, in volts over
        meter.
        
        See also
        --------
        Dielectric.compute_field, ConductorSet.compute_field
        """
        where = np.array([x, y])
        where_shape = (1,) * (len(where.shape) - 1)
        
        l = self.lengths
        l = 1/2 * np.array([-l, l])
        l =           l.reshape(2, 1, *where_shape,    -1)
        m = self.slopes
        mx, my =   m.reshape(2, 1, 1, *where_shape,    -1)
        vec_m =       m.reshape(1, 2, *where_shape,    -1)
        c = self.centers
        cx, cy =   c.reshape(2, 1, 1, *where_shape,    -1)
        vec_c =       c.reshape(1, 2, *where_shape,    -1)
        x, y = where.reshape(2, 1, 1, *where.shape[1:], 1)
        vec_x =   where.reshape(1, *where.shape,        1)

        integrals = numexpr.evaluate('-vec_m / (2 * (mx**2 + my**2)) * log((mx*l - (x-cx))**2 + (my*l - (y-cy))**2) +\
        where(\
            mx * (y - cy) - my * (x - cx) != 0,\
            (vec_x - vec_c + vec_m * (mx*(cx-x) + my*(cy-y)) / (mx**2 + my**2)) /\
            abs(mx * (y - cy) - my * (x - cx)) *\
            arctan(((mx**2 + my**2)*l + mx*(cx-x) + my*(cy-y)) / abs(mx * (y - cy) - my * (x - cx))),\
            1 / (mx ** 2 + my ** 2) *\
            (vec_m * (mx * (cx - x) + my * (cy - y)) + (mx ** 2 + my ** 2) * (vec_x - vec_c)) /\
            ((mx ** 2 + my ** 2) * l + mx * (cx - x) + my * (cy - y))\
        )')
        
        field = np.sum((integrals[1] - integrals[0]) * self.sigmas.reshape(1, *where_shape, -1), axis=-1) / (2 * np.pi * constants.epsilon_0)
        
        assert field.shape == where.shape
        
        return field
        
class CircleConductor(Conductor):
    """
    Convenience class for constructing a circular conductor.
    
    Arguments
    ---------
    center : pair of numbers
        The x, y coordinates of the center of the circle, in meters.
    radius : number
        The radius of the circle, in meters.
    segments : integer of 1D array
        If integer: the number of equal segments of the polygon that
        approximates the circle. If array: the values in the array are the
        angles, in radians, at which the vertices of the polygon are located.
    
    Keyword arguments
    -----------------
    Additional keyword arguments are passed to Conductor.
    """
    def __init__(self, center=(0, 0), radius=1, segments=12, **kw):
        if isinstance(segments, int):
            angles = np.linspace(0, 2 * np.pi, segments + 1)[:-1]
        else:
            angles = np.asarray(segments)
            assert len(angles.shape) == 1
            assert len(angles) >= 2
        vertexes_x = center[0] + radius * np.cos(angles)
        vertexes_y = center[1] + radius * np.sin(angles)
        super(CircleConductor, self).__init__(vertexes_x, vertexes_y, closed=True, **kw)

class SegmentConductor(Conductor):
    """
    Convenience class for constructing a conductor segment.
    
    Arguments
    ---------
    endpoint_A : pair of numbers
    endpoint_B : pair of numbers
        The x, y coordinates of the endpoints of the segment, in meters.
    segments : integer of 1D array
        If integer: the number of subsegments into which the conductor is
        divided. If array: the values in the array are the positions of the
        subsegments vertices, relative to the length of the segment, going from
        A to B. A and B correspond to the first and last element of the array.
    center : pair of numbers
        The x, y coordinates of the center of the segment, in meters.
    vector_A_to_B : pair of numbers
        The x, y components of the the vector going from A to B, i.e. the
        coordinates of B relative to A.
    
    The geometrical information specified by endpoint_A, endpoint_B, center and
    vector_A_to_B is redundant. They are used with the following rules:
      * If both endpoint_A and endpoint_B are specified, center and
        vector_A_to_B are ignored.
      * Otherwise, if vector_A_to_B and one of endpoint_A, endpoint_B or
        center, checked in this order, are specified, the remaining arguments
        are ignored.
      * In other cases a ValueError is raised, even if center and one of
        endpoint_A or endpoint_B would be sufficient.
    
    Keyword arguments
    -----------------
    Additional keyword arguments are passed to Conductor.
    """
    def __init__(self, endpoint_A=None, endpoint_B=None, segments=10, center=None, vector_A_to_B=None, **kw):
        if isinstance(segments, int):
            steps = np.linspace(0, 1, segments + 1)
        else:
            steps = np.asarray(segments)
            assert len(steps.shape) == 1
            assert len(steps) >= 2
        
        if not (endpoint_A is None) and not (endpoint_B is None):
            pass
        elif not (endpoint_A is None) and not (vector_A_to_B is None):
            endpoint_B = (
                endpoint_A[0] + vector_A_to_B[0],
                endpoint_A[1] + vector_A_to_B[1]
            )
        elif not (endpoint_B is None) and not (vector_A_to_B is None):
            endpoint_A = (
                endpoint_B[0] - vector_A_to_B[0],
                endpoint_B[1] - vector_A_to_B[1]
            )
        elif not (center is None) and not (vector_A_to_B is None):
            endpoint_A = (
                center[0] - vector_A_to_B[0] / 2,
                center[1] - vector_A_to_B[1] / 2,
            )
            endpoint_B = (
                center[0] + vector_A_to_B[0] / 2,
                center[1] + vector_A_to_B[1] / 2,
            )
        else:
            raise ValueError('insufficient geometrical information')
        
        vertexes_x = endpoint_A[0] + (endpoint_B[0] - endpoint_A[0]) * steps
        vertexes_y = endpoint_A[1] + (endpoint_B[1] - endpoint_A[1]) * steps
        
        super(SegmentConductor, self).__init__(vertexes_x, vertexes_y, closed=False, **kw)

class RectangleConductor(Conductor):
    """
    Convenience class for constructing a conductor rectangle. The rectangle is
    aligned with the x and y axes.
    
    Arguments
    ---------
    bottom_left : pair of numbers
        The x, y coordinates of the bottom left vertex of the rectangle, in
        meters.
    sides : pair of numbers
        The horizontal and vertical lengths of the sides of the rectangle, in
        meters.
    segments : pair of numbers
        The number of subsegments the horizontal and vertical sides are divided
        into.
    
    Keyword arguments
    -----------------
    Additional keyword arguments are passed to Conductor.
    """
    def __init__(self, bottom_left=(0, 0), sides=(1, 1), segments=(10, 10), **kw):
        vertexes_x = np.concatenate([
            np.linspace(bottom_left[0], bottom_left[0] + sides[0], segments[0] + 1)[:-1],
            np.ones(segments[1]) * (bottom_left[0] + sides[0]),
            np.linspace(bottom_left[0] + sides[0], bottom_left[0], segments[0] + 1)[:-1],
            np.ones(segments[1]) * bottom_left[0]
        ])
        vertexes_y = np.concatenate([
            np.ones(segments[0]) * bottom_left[1],
            np.linspace(bottom_left[1], bottom_left[1] + sides[1], segments[1] + 1)[:-1],
            np.ones(segments[0]) * (bottom_left[1] + sides[1]),
            np.linspace(bottom_left[1] + sides[1], bottom_left[1], segments[1] + 1)[:-1]
        ])
        assert len(vertexes_x) == len(vertexes_y) == 2 * (segments[0] + segments[1])
        super(RectangleConductor, self).__init__(vertexes_x, vertexes_y, closed=True, **kw)
        
class Dielectric(object):
    """
    Class to represent dielectrics. A dielectric is a collection of rectangles
    with sides aligned with the x, y axes; on each rectangle there is a uniform
    polarization density.
    
    Arguments
    ---------
    bottom_left_x : array with shape (N,)
    bottom_left_y : array with shape (N,)
        The x, y coordinates of the bottom left vertices of the rectangles, in
        meters.
    width : array with shape (N,)
    height : array with shape (N,)
        The lengths of the sides of the rectangles, in meters.
    epsilon_rel : number
        The relative dielectric permittivity.
    name : string
        Label used in plots.
    
    Methods
    -------
    draw :
        Draw the rectangles on a matplotlib axis.
    compute_potential :
        Compute the potential generated by the dielectric at specified points.
    compute_field :
        Compute the field generated by the dielectric at specified points.
    
    Operations
    ----------
    + :
        The sum of two dielectrics is a dielectric with all the rectangles of
        the two. The name and dielectric permittivity are inherited from the
        first addend.
    
    Properties
    ----------
    name : string
        The name given at initialization.
    centers : array of shape (2, N)
        The x, y coordinates of the centers of the rectangles, in meters.
    areas : array of shape (N,)
        The areas of the rectangles, in meters squared.
    bottom_left : array of shape (2, N)
        The x, y coordinates of the bottom left vertices of the rectangles, in
        meters.
    sides : array of shape (2, N)
        The lengths of the sides of the rectangles, in meters.
    polarizability : number
        The polarizability of the dielectric, i.e. epsilon_rel - 1.
    Ps : None or array of shape (N,)
        The polarization densities of the rectangles, in coulomb over meter
        squared. It is None if the dielectric is not in a solved ConductorSet.
    polarization_per_unit_length : number
        The polarization density along the z axis, in coulomb.
    
    See also
    --------
    RectangleDielectric
    """
    def __init__(self, bottom_left_x, bottom_left_y, width, height, epsilon_rel=1, name='dielectric'):
        bottom_left_x = np.asarray(bottom_left_x)
        bottom_left_y = np.asarray(bottom_left_y)
        width = np.asarray(width)
        height = np.asarray(height)
        
        assert len(bottom_left_x.shape) == 1
        assert len(bottom_left_y.shape) == 1
        assert len(width.shape) == 1
        assert len(height.shape) == 1
        assert len(bottom_left_x) == len(bottom_left_y) == len(width) == len(height) > 0
        
        self._c = np.array([bottom_left_x, bottom_left_y])
        self._L = np.array([width, height])
        
        self._chi = float(epsilon_rel) - 1
        self._name = str(name)
        
        self._Ps = None
    
    def __add__(self, obj):
        if not isinstance(obj, Dielectric):
            raise TypeError("unsupported operand type(s) for +: '{}' and '{}'".format(type(self), type(obj)))
        bottom_left = np.concatenate([self.bottom_left, obj.bottom_left], axis=1)
        sides = np.concatenate([self.sides, obj.sides], axis=1)
        
        # How to add epsilon_rel and name?
        # maybe support different epsilon_rel for each piece of the dielectric;
        # comes at no computational cost.
        epsilon_rel = 1 + self.polarizability
        name = self.name
        
        return Dielectric(bottom_left[0], bottom_left[1], sides[0], sides[1], epsilon_rel, name)
    
    @property
    def name(self):
        return self._name
    
    @property
    def centers(self):
        return self._c + self._L / 2
    
    @property
    def areas(self):
        return np.prod(self._L, axis=0)
    
    @property
    def bottom_left(self):
        return self._c
    
    @property
    def sides(self):
        return self._L
    
    @property
    def polarizability(self):
        return self._chi
    
    @property
    def Ps(self):
        return self._Ps
    
    @property
    def polarization_per_unit_length(self):
        return np.sum(self.Ps * self.areas.reshape(1, -1), axis=1)

    def draw(self, ax=None, **kw):
        """
        Draw the rectangles on a matplotlib axis.
        
        Arguments
        ---------
        ax : None or matplotlib axis
            If None, use the current axis.
        
        Keyword arguments
        -----------------
        Additional keyword arguments are passed to ax.plot, taking precedence
        over the defaults.
        
        Returns
        -------
        The return value of ax.plot.
        
        See also
        --------
        Conductor.draw, ConductorSet.draw
        """
        if ax is None:
            ax = plt.gca()
        kwargs = dict(
            label='{:s} $\\varepsilon$ = {:.4g}'.format(self._name, 1 + self._chi)
        )
        kwargs.update(kw)
        
        l = self._c
        r = self._c + self._L
        
        x = np.array([l[0], l[0], r[0], r[0], l[0]])
        y = np.array([l[1], r[1], r[1], l[1], l[1]])
        
        lines = []
        line, = ax.plot(x[:,0], y[:,0], **kwargs)
        kwargs.update(color=line.get_color(), label=None)
        lines.append(line)
        for i in range(1, x.shape[1]):
            lines.append(ax.plot(x[:,i], y[:,i], **kwargs)[0])
        return lines
    
    def compute_potential(self, x, y):
        """
        Compute the potential generated by the dielectric at specified points.
        
        Arguments
        ---------
        x, y: arrays with same shape
            x and y coordinates of the points, in meters.
        
        Returns
        -------
        Array of potentials, in volts, with same shape as x and y.
        
        See also
        --------
        Conductor.compute_potential, ConductorSet.compute_potential
        """
        where = np.array([x, y])
        where_shape = (1,) * (len(where.shape) - 1)
            
        uv = np.array([
            self.bottom_left.reshape(2, *where_shape, -1) - where.reshape(*where.shape, 1),
            self.bottom_left.reshape(2, *where_shape, -1) - where.reshape(*where.shape, 1) + self.sides.reshape(2, *where_shape, -1)
        ])
        u = uv[:,0]
        v = uv[:,1]
    
        f = lambda u, v: numexpr.evaluate('-1/2 * (v * log(u**2 + v**2) + 2*u * arctan(v/u) - 2*v)')
        delta = lambda f, u, v: f(u[1], v[1]) - f(u[1], v[0]) - f(u[0], v[1]) + f(u[0], v[0])
        integrals_x = delta(f, u, v)
        integrals_y = delta(f, v, u)
    
        integrals = np.array([integrals_x, integrals_y])
        
        potential = np.sum(integrals * self.Ps.reshape(2, *where_shape, -1), axis=(0,-1)) / (2 * np.pi * constants.epsilon_0)
        
        assert potential.shape == where.shape[1:]
        
        return potential
    
    def compute_field(self, x, y):
        """
        Compute the electric field generated by the dielectric at specified
        points.
        
        Arguments
        ---------
        x, y: arrays with shape S
            x and y coordinates of the points, in meters.
        
        Returns
        -------
        Array with shape (2, *S) of field x and y components, in volts over
        meter.
        
        See also
        --------
        Conductor.compute_field, ConductorSet.compute_field
        """
        where = np.array([x, y])
        where_shape = (1,) * (len(where.shape) - 1)
            
        uv = np.array([
            self.bottom_left.reshape(2, *where_shape, -1) - where.reshape(*where.shape, 1),
            self.bottom_left.reshape(2, *where_shape, -1) - where.reshape(*where.shape, 1) + self.sides.reshape(2, *where_shape, -1)
        ])
        u = uv[:,0]
        v = uv[:,1]
    
        f = lambda u, v: numexpr.evaluate('-arctan(v / u)')
        g = lambda u, v: numexpr.evaluate('-1/2 * log(u**2 + v**2)')
        delta = lambda f, u, v: f(u[1], v[1]) - f(u[1], v[0]) - f(u[0], v[1]) + f(u[0], v[0])
        integrals_Px_x = delta(f, u, v)
        integrals_Py_x = delta(g, u, v)
        integrals_Px_y = integrals_Py_x # because symmetric in u, v
        integrals_Py_y = delta(f, v, u)
        
        integrals = np.array([
            [integrals_Px_x, integrals_Py_x],
            [integrals_Px_y, integrals_Py_y]
        ])
        
        field = np.sum(integrals * self.Ps.reshape(1, 2, *where_shape, -1), axis=(1,-1)) / (2 * np.pi * constants.epsilon_0)
            
        assert field.shape == where.shape
        
        return field

class RectangleDielectric(Dielectric):
    """
    Convenience class for constructing a rectangular dielectric.
    
    Arguments
    ---------
    bottom_left : pair of numbers
        x, y coordinates of the bottom left vertex of the rectangle, in meters.
    sides : pair of numbers
        Lenghts of the horizontal and vertical sides of the rectangle, in
        meters.
    segments : pair of integers
        Number of subdivisions along x and y directions; the rectangle is
        divided in smaller rectangles.
    """
    def __init__(self, bottom_left=(0, 0), sides=(1, 1), segments=(10, 10), **kw):
        bottom_left_x = bottom_left[0] + sides[0] * np.arange(segments[0]) / segments[0]
        bottom_left_y = bottom_left[1] + sides[1] * np.arange(segments[1]) / segments[1]

        base = np.ones((segments[0], segments[1]))
        bottom_left_x = base * bottom_left_x.reshape(-1, 1)
        bottom_left_y = base * bottom_left_y.reshape(1, -1)

        width = base * sides[0] / segments[0]
        height = base * sides[1] / segments[1]
        
        super(RectangleDielectric, self).__init__(bottom_left_x.flatten(), bottom_left_y.flatten(), width.flatten(), height.flatten(), **kw)        

class ConductorSet(object):
    """
    Object to bring together conductors and dielectrics and solve the system.
    After calling solve(), one can call the methods compute_potential,
    compute_field both on the ConductorSet object and on the conductors and
    dielectrics inside it, and the methods draw_field and draw_potential on the
    ConductorSet object.
    
    Arguments
    ---------
    The arguments are Conductor and Dielectric instances. If an object with
    another class if given, ValueError is raised.
    
    Methods
    -------
    solve :
        Solve the system, determining charge densities of the conductors and
        polarization densities of the dielectrics.
    draw :
        Draw dielectrics and conductors.
    draw_potential :
        Draw the potential.
    draw_field:
        Draw the electric field.
    compute_potential :
        Compute the potential at given points.
    compute_field:
        Compute the electric field at given points.
    
    Properties
    ----------
    conductors : tuple
        Conductors given at initialization.
    dielectrics : tuple
        Dielectrics given at initialization.
    potential_offset : number
        Potential offset computed by solve() to impose zero potential at
        infinity.
    """
    def __init__(self, *conductors_and_dielectrics):
        self._conductors = []
        self._dielectrics = []
        for obj in conductors_and_dielectrics:
            if isinstance(obj, Conductor):
                self._conductors.append(obj)
            elif isinstance(obj, Dielectric):
                self._dielectrics.append(obj)
            else:
                raise ValueError('unrecognized object')
        self._conductors = tuple(self._conductors)
        self._dielectrics = tuple(self._dielectrics)
        assert len(self._conductors) > 0
    
    @property
    def conductors(self):
        return self._conductors
    
    @property
    def dielectrics(self):
        return self._dielectrics
    
    def draw(self, *args, **kw):
        """
        Draw dielectrics and conductors by calling draw() on each dielectric
        and conductor object. All arguments are passed down unchanged.
        Dielectrics are drawn before conductors.
        
        Returns
        -------
        Returns the concatenation of the return values of the calls to draw()
        on the objects.
        """
        rt = []
        for dielectric in self.dielectrics:
            rt += dielectric.draw(*args, **kw)
        for conductor in self.conductors:
            rt += conductor.draw(*args, **kw)
        return rt
    
    def draw_potential(self, x, y, ax=None, use_conductors=True, use_dielectrics=True, **kw):
        """
        Draw the potential as a pixel colormap.
        
        Arguments
        ---------
        x, y : 1D arrays
            x, y coordinates of the pixel grid, in meters. x and y are the
            coordinates of the pixel *edges*, the potential is computed in the
            pixel *centers*.
        ax : None or matplotlib axis
            Axis to draw onto. If None, use current axis.
        use_conductors : boolean
        use_dielectrics : boolean
            Flags to ignore conductors or dielectrics when computing the
            potential. Note that the charge configuration is determined by
            calling solve(), not by this options, so ignoring the contribute of
            the dielectrics will not eliminate their effect on the charges of
            the conductors.
        
        Keyword arguments
        -----------------
        Keyword arguments are passed to ax.pcolormesh, taking precedence over
        defaults.
        
        Returns
        -------
        Returns the return value of ax.pcolormesh.
        """
        if ax is None:
            ax = plt.gca()
        
        X, Y = np.meshgrid(x, y)
        x_compute = (x[1:] + x[:-1]) / 2
        y_compute = (y[1:] + y[:-1]) / 2
        X_compute, Y_compute = np.meshgrid(x_compute, y_compute)
        
        potential = self.compute_potential(X_compute, Y_compute, use_conductors=use_conductors, use_dielectrics=use_dielectrics)
        
        kwargs = dict(
            cmap='gray',
            label='potential'
        )
        kwargs.update(kw)

        return ax.pcolormesh(X, Y, potential, **kwargs), potential

        
    def draw_field(self, x, y, ax=None, use_conductors=True, use_dielectrics=True, scale=None, **kw):
        """
        Draw the electric field as arrows.
        
        Arguments
        ---------
        x, y : arrays with same size
            x, y coordinates of the points where the field is computed and
            plotted. The arrays are flattened before use.
        ax : None or matplotlib axis
            Axis to draw onto. If None, use current axis.
        use_conductors : boolean
        use_dielectrics : boolean
            Flags to ignore conductors or dielectrics when computing the
            potential. Note that the charge configuration is determined by
            calling solve(), not by this options, so ignoring the contribute of
            the dielectrics will not eliminate their effect on the charges of
            the conductors.
        scale : None or one of 'linear', 'log', 'uniform'
            None and 'linear' means the length of the arrow is proportional to
            the field modulus. 'log' means it is proportional to the logarithm
            of the field, where the logarithm yields zero on the minimum field
            modulus computed. 'uniform' means all arrows have the same length
            apart from zero field points; only direction information is
            preserved. Other values raise a KeyError.
        
        Keyword arguments
        -----------------
        Keyword arguments are passed to ax.quiver, taking precedence over
        defaults.
        
        Returns
        -------
        Returns the return value of ax.quiver.
        """
        if ax is None:
            ax = plt.gca()
        
        # flattening is needed for quiver()
        x = np.asarray(x).reshape(-1)
        y = np.asarray(y).reshape(-1)
        
        assert x.shape == y.shape
        
        field = self.compute_field(x, y, use_conductors=use_conductors, use_dielectrics=use_dielectrics)
        if scale is None or scale == 'linear':
            pass
        else:
            norm = np.sqrt(np.sum(field ** 2, axis=0))
            norm[norm == 0] = 1
            field /= norm
            if scale == 'uniform':
                pass
            elif scale == 'log':
                # this logaritmic scale should not end at zero because it cancels direction information
                field *= np.log(norm / np.min(norm))
            else:
                raise KeyError(scale)

        kwargs = dict(
            color='red',
            label='electric field'
        )
        kwargs.update(kw)
        
        return ax.quiver(x, y, *field, **kwargs), field
        
    def solve(self, zero_potential_at_infinity=True, use_dielectrics=True, verbose=True):
        """
        Solve the electrostatic problem, computing the charge surface densities
        on the conductor elementary segments and the polarization volume
        densities on the dielectric elementary rectangles.
        
        After calling solve(), one can call compute_potential, compute_field,
        draw_potential, draw_field and access the property potential_offset.
        
        Arguments
        ---------
        zero_potential_at_infinity : boolean
            If True: the potentials are left free to translate to have zero
            potential at infinity, which is equivalent to have null total
            charge linear density. If False: potentials are left as specified
            in the conductors, in this case the potential at infinity will
            diverge with the sign depending on the sign of the total charge
            linear density. Note that in any case the <potential> properties of
            the conductor objects are not changed, the potential offset can be
            retrieved from the property <potential_offset> of the ConductorSet
            object. <potential_offset> will yield zero if
            zero_potential_at_infinity=False.
        use_dielectrics : boolean
            Flag to ignore dielectrics in the computation. Having also a flag
            to ignore conductors would not make sense because the solution is
            trivial.
        verbose : boolean
            If True, print information during the computation.
        """
        epsilon_0 = 1 #constants.epsilon_0
        
        printer = print if verbose else lambda *args: None
        
        printer('***** ConductorSet.solve *****')
        
        # extract conductor properties
        cond_shapes = np.array([len(conductor.lengths) for conductor in self.conductors])
        cond_potentials = np.concatenate([
            np.ones(shape) * conductor.potential
            for shape, conductor in zip(cond_shapes, self.conductors)
        ])
        cond_slopes = np.concatenate([conductor.slopes for conductor in self.conductors], axis=1)
        cond_centers = np.concatenate([conductor.centers for conductor in self.conductors], axis=1)
        cond_lengths = np.concatenate([conductor.lengths for conductor in self.conductors])
        N_cond = len(cond_potentials)
                
        assert len(cond_slopes[0]) == len(cond_centers[0]) == len(cond_lengths) == len(cond_potentials)

        printer('{:d} conductor objects for a total of {:d} segments'.format(len(cond_shapes), N_cond))

        # extract dielectric properties
        if len(self.dielectrics) == 0:
            use_dielectrics = False
        if use_dielectrics:
            diel_shapes = np.array([len(dielectric.areas) for dielectric in self.dielectrics])
            diel_chis = np.concatenate([
                np.ones(shape) * dielectric.polarizability
                for shape, dielectric in zip(diel_shapes, self.dielectrics)
            ])
            diel_centers = np.concatenate([dielectric.centers for dielectric in self.dielectrics], axis=1)
            diel_bottom_left = np.concatenate([dielectric.bottom_left for dielectric in self.dielectrics], axis=1)
            diel_sides = np.concatenate([dielectric.sides for dielectric in self.dielectrics], axis=1)
            N_diel = len(diel_chis)
            
            assert len(diel_chis) == len(diel_centers[0]) == len(diel_bottom_left[0]) == len(diel_sides[0])
            
            printer('{:d} dielectric objects for a total of {:d} rectangles'.format(len(diel_shapes), N_diel))
        else:
            N_diel = 0
            
            printer('no dielectrics')
        
        
        # linear system to solve is Ax=B
        # layout of the equations:
        # *––––––––––––*––––––––––––*   *–––––––––––*
        # | cond<-cond | cond<-diel |   | potential |
        # |––––––––––––|––––––––––––|   |–––––––––––|
        # | diel<-cond | diel<-diel | = |     0     |
        # |––––––––––––|––––––––––––|   |–––––––––––|
        # | sum charge |            |   |     0     |
        # *––––––––––––*––––––––––––*   *–––––––––––*
        # 
        # arrangement of unknowns:
        # [*sigma, *P_x, *P_y, logr0]
        
        # construct B
        B_potential = cond_potentials
        B_polarization = np.zeros(2 * N_diel if use_dielectrics else 0)
        B_boundary_conditions = np.zeros(1 if zero_potential_at_infinity else 0)
        B = np.concatenate([
            B_potential,
            B_polarization,
            B_boundary_conditions
        ])
        
        # construct A_cc
        printer('computing conductor->conductor coefficients...')
        
        # careful: l is used also in A_dc
        l = 1/2 * np.array([-cond_lengths, cond_lengths]).reshape(2, 1, -1)
        mx, my =                           cond_slopes.reshape(2, 1, 1, -1)
        cx, cy =                          cond_centers.reshape(2, 1, 1, -1)
        x, y =                            cond_centers.reshape(2, 1, -1, 1)

        A_cc = numexpr.evaluate('-1/2 * (\
        2 / (mx ** 2 + my ** 2) * abs(mx * (y - cy) - my * (x - cx)) *\
        arctan(((mx ** 2 + my ** 2) * l + mx * (cx - x) + my * (cy - y)) / abs(mx * (y - cy) - my * (x - cx))) -\
        2 * l +\
        ((mx * (cx - x) + my * (cy - y)) / (mx ** 2 + my ** 2) + l) *\
        log((mx * l - (x - cx)) ** 2 + (my * l - (y - cy)) ** 2))')

        assert A_cc.shape == (2, N_cond, N_cond)
        
        # using np.diff(A_cc, axis=0) is 2x slower!
        A_cc = (A_cc[1] - A_cc[0]) / (2 * np.pi * epsilon_0)
        
        if use_dielectrics:
            # construct A_dc
            printer('computing conductor->dielectric coefficients...')
        
            l =                    l.reshape(2, 1,  1, -1)
            mx, my =  cond_slopes.reshape(2, 1, 1,  1, -1)
            vec_m =      cond_slopes.reshape(1, 2,  1, -1)
            cx, cy = cond_centers.reshape(2, 1, 1,  1, -1)
            vec_c =     cond_centers.reshape(1, 2,  1, -1)
            x, y =   diel_centers.reshape(2, 1, 1, -1,  1)
            vec_x =     diel_centers.reshape(1, 2, -1,  1)

            A_dc = numexpr.evaluate('-vec_m / (2 * (mx**2 + my**2)) * log((mx*l - (x-cx))**2 + (my*l - (y-cy))**2) +\
            where(\
                mx * (y - cy) - my * (x - cx) != 0,\
                (vec_x - vec_c + vec_m * (mx*(cx-x) + my*(cy-y)) / (mx**2 + my**2)) /\
                abs(mx * (y - cy) - my * (x - cx)) *\
                arctan(((mx**2 + my**2)*l + mx*(cx-x) + my*(cy-y)) / abs(mx * (y - cy) - my * (x - cx))),\
                1 / (mx ** 2 + my ** 2) *\
                (vec_m * (mx * (cx - x) + my * (cy - y)) + (mx ** 2 + my ** 2) * (vec_x - vec_c)) /\
                ((mx ** 2 + my ** 2) * l + mx * (cx - x) + my * (cy - y))\
            )')
        
            assert A_dc.shape == (2, 2, N_diel, N_cond)
        
            A_dc = (A_dc[1] - A_dc[0]) * diel_chis.reshape(1, -1, 1) / (2 * np.pi)
            A_dc = A_dc.reshape(2 * N_diel, N_cond)
        
            # construct A_cd
            printer('computing dielectric->conductor coefficients...')
        
            uv = np.array([
                diel_bottom_left.reshape(2, -1, 1) - cond_centers.reshape(2, 1, -1),
                diel_bottom_left.reshape(2, -1, 1) - cond_centers.reshape(2, 1, -1) + diel_sides.reshape(2, -1, 1)
            ])
            u = uv[:,0]
            v = uv[:,1]
        
            f = lambda u, v: numexpr.evaluate('-1/2 * (v * log(u**2 + v**2) + 2*u * arctan(v/u) - 2*v)')
            delta = lambda f, u, v: f(u[1], v[1]) - f(u[1], v[0]) - f(u[0], v[1]) + f(u[0], v[0])
            A_cd_x = delta(f, u, v)
            A_cd_y = delta(f, v, u)
        
            A_cd = np.array([A_cd_x, A_cd_y])
            A_cd /= 2 * np.pi * epsilon_0
            A_cd = A_cd.reshape(2 * N_diel, N_cond).transpose()
            # We did the transposition at the end because, if done before, flattening x and y would have been more complicated.
        
            # construct A_dd
            printer('computing dielectric->dielectric coefficients...')
        
            uv = np.array([
                diel_bottom_left.reshape(2, -1, 1) - diel_centers.reshape(2, 1, -1),
                diel_bottom_left.reshape(2, -1, 1) - diel_centers.reshape(2, 1, -1) + diel_sides.reshape(2, -1, 1)
            ])
            u = uv[:,0]
            v = uv[:,1]
        
            f = lambda u, v: numexpr.evaluate('-arctan(v / u)')
            g = lambda u, v: numexpr.evaluate('-1/2 * log(u**2 + v**2)')
            A_dd_Px_x = delta(f, u, v) * diel_chis.reshape(1, -1)
            A_dd_Py_x = delta(g, u, v) * diel_chis.reshape(1, -1)
            A_dd_Px_y = A_dd_Py_x # because symmetric in u, v
            A_dd_Py_y = delta(f, v, u) * diel_chis.reshape(1, -1)
        
            A_dd_Px = np.concatenate([A_dd_Px_x, A_dd_Px_y], axis=1)
            A_dd_Py = np.concatenate([A_dd_Py_x, A_dd_Py_y], axis=1)
            A_dd = np.concatenate([A_dd_Px, A_dd_Py], axis=0).transpose()
            A_dd /= 2 * np.pi
            np.fill_diagonal(A_dd, np.diagonal(A_dd) - 1)
        
            assert A_dd.shape == (2 * N_diel, 2 * N_diel)
        else:
            A_cd = np.zeros((N_cond, 0))
            A_dc = np.zeros((0, N_cond))
            A_dd = np.zeros((0, 0))
        
        if zero_potential_at_infinity:
            # construct A_offset
            A_offset = np.ones((N_cond, 1)) * np.sum(cond_lengths) / (2 * np.pi * epsilon_0)
        
            # construct A_charge
            A_charge = cond_lengths.reshape(1, -1)
            
            # various zeroes of A
            A_offset_diel = np.zeros((2 * N_diel, 1))
            A_charge_diel = np.zeros((1, 2 * N_diel))
            A_charge_offset = np.zeros((1, 1))
        else:
            A_offset = np.zeros((N_cond, 0))
            A_charge = np.zeros((0, N_cond))
            A_offset_diel = np.zeros((2 * N_diel, 0))
            A_charge_diel = np.zeros((0, 2 * N_diel))
            A_charge_offset = np.zeros((0, 0))
        
        A_cc_cd_offset = np.concatenate([A_cc, A_cd, A_offset], axis=1)
        A_dc_dd = np.concatenate([A_dc, A_dd, A_offset_diel], axis=1)
        A_charge = np.concatenate([A_charge, A_charge_diel, A_charge_offset], axis=1)
        A = np.concatenate([A_cc_cd_offset, A_dc_dd, A_charge], axis=0)
        
        assert A.shape == (len(B), len(B))
        
        printer('solving linear system...')
        solution = linalg.solve(A, B)
        
        sigmas = solution[:N_cond] * constants.epsilon_0 / epsilon_0
        if use_dielectrics:
            Ps = np.array([
                solution[N_cond:N_cond + N_diel],
                solution[N_cond + N_diel:N_cond + 2 * N_diel]
            ]) * constants.epsilon_0 / epsilon_0
        if zero_potential_at_infinity:
            logr0 = solution[-1]
        else:
            logr0 = 0
        
        idxs = np.cumsum(np.concatenate([[0], cond_shapes]))
        for i in range(len(self.conductors)):
            self.conductors[i]._sigmas = sigmas[idxs[i]:idxs[i+1]]
       
        if use_dielectrics:
            idxs = np.cumsum(np.concatenate([[0], diel_shapes]))
            for i in range(len(self.dielectrics)):
                self.dielectrics[i]._Ps = Ps[idxs[i]:idxs[i+1]]
        
        self._potential_offset = -np.sum(cond_lengths) * logr0 / (2 * np.pi * epsilon_0)
        
        printer('************ done ************')
    
    @property
    def potential_offset(self):
        return self._potential_offset
    
    def compute_potential(self, x, y, use_conductors=True, use_dielectrics=True):
        potential = np.sum([
            obj.compute_potential(x, y)
            for obj in (self.conductors if use_conductors else ()) + (self.dielectrics if use_dielectrics else ())
        ], axis=0)
        return potential
    
    def compute_field(self, x, y, use_conductors=True, use_dielectrics=True):
        field = np.sum([
            obj.compute_field(x, y)
            for obj in (self.conductors if use_conductors else ()) + (self.dielectrics if use_dielectrics else ())
        ], axis=0)
        return field
